import matplotlib.pyplot as plt
import matplotlib.patches as patches

# Create a figure and axis
fig, ax = plt.subplots(figsize=(8, 10))

# Define colors
tile_clear_color = 'green'
tile_occupied_color = 'red'
robot_white_color = 'white'
robot_black_color = 'black'
text_color_black = 'black'
text_color_white = 'white'

# Define grid size
rows, cols = 5, 4
tile_size = 1

# Draw tiles
for row in range(rows):
    for col in range(cols):
        # Calculate position
        x = col * tile_size
        y = (rows - 1 - row) * tile_size
        
        # Determine if tile is clear or occupied
        if (row, col) in [(1, 0), (1, 1)]:
            tile_color = tile_occupied_color
        else:
            tile_color = tile_clear_color
        
        # Create a rectangle for each tile
        tile = patches.Rectangle((x, y), tile_size, tile_size, linewidth=1, edgecolor='black', facecolor=tile_color)
        ax.add_patch(tile)
        
        # Add text label for tile
        tile_label = f'tile_{row}-{col+1}'
        ax.text(x + tile_size/2, y + tile_size/2, tile_label, fontsize=8, ha='center', va='center', color=text_color_black)

# Draw robots
robots = {
    'robot1': {'position': (1, 0), 'color': robot_white_color},
    'robot2': {'position': (1, 1), 'color': robot_black_color}
}

for robot, info in robots.items():
    row, col = info['position']
    x = col * tile_size + tile_size/2
    y = (rows - 1 - row) * tile_size + tile_size/2
    
    # Create a circle for each robot
    robot_circle = patches.Circle((x, y), tile_size/4, facecolor=info['color'], edgecolor='black')
    ax.add_patch(robot_circle)
    
    # Add text label for robot
    robot_label = f'{robot}\n{info["color"]}'
    ax.text(x, y, robot_label, fontsize=8, ha='center', va='center', color=text_color_black if info['color'] == robot_white_color else text_color_white)

# Add available colors
available_colors = {
    'color_white': (robot_white_color, (5, 4)),
    'color_black': (robot_black_color, (5, 3))
}

for color_name, (color, position) in available_colors.items():
    x, y = position
    color_rect = patches.Rectangle((x, y), tile_size, tile_size/2, linewidth=1, edgecolor='black', facecolor=color)
    ax.add_patch(color_rect)
    ax.text(x + tile_size/2, y + tile_size/4, color_name, fontsize=8, ha='center', va='center', color=text_color_black if color == robot_white_color else text_color_white)

# Add legend
legend_elements = [
    patches.Patch(facecolor=tile_clear_color, edgecolor='black', label='Clear Tile'),
    patches.Patch(facecolor=tile_occupied_color, edgecolor='black', label='Occupied Tile'),
    patches.Patch(facecolor=robot_white_color, edgecolor='black', label='Robot Holding White'),
    patches.Patch(facecolor=robot_black_color, edgecolor='black', label='Robot Holding Black')
]

ax.legend(handles=legend_elements, loc='upper right', bbox_to_anchor=(1.2, 1), fontsize=8)

# Set limits and hide axes
ax.set_xlim(-1, cols + 2)
ax.set_ylim(-1, rows + 2)
ax.set_aspect('equal')
ax.axis('off')

# Save the figure
plt.savefig('tiles/one_shot/ini_diagram_code/attempts/attempt_6.png', bbox_inches='tight')
plt.show()